%% Script that processes the Head Tracking data
clear,clc,close all

%% Variables initialization
RightEar_NH  = [2,5,12,15];         % NH participants with bad RIGHT ear
RightEar_UHL = [2,5,12,15,16];      % UHL participants with bad RIGHT ear
edges = -125:2.5:75;
fs = 4410;
DEBUG = 0;
Excluded = [104,1;104,2;104,3;104,4;104,5;104,6;108,1;108,2;108,3;...
    109,1;112,4;203,5;208,1;208,2;208,3;208,4;208,5;208,6;210,1;210,2;...
    210,3;211,1;211,2;211,3;212,1;212,2;212,3;105,4;113,4;116,3;213,1;...
    213,4;214,4;216,1;207,1;209,1;215,4;102,1];

%% Build a matrix of histograms per participant
L_hist = length(edges)-1;
H = nan(32,L_hist);
STD = nan(32*6,2);
fs_orig = 44100;
Sbjs = [201:216 , 101:116];         % Loop in NH and UHL participants
for i=1:32
    % Estimate individual histograms
    h_ind = nan(6,L_hist);          % Initialise variable
    clf(figure(2)),hold on
    for j=1:6                       % Loop in recordings (x6)

        % Check data availability
        pos_sbj = find(Excluded(:,1)-Sbjs(i) == 0);
        pos_run = find(Excluded(pos_sbj,2)-j == 0);
        
        if isempty(pos_run)
            % Processing
            fprintf(sprintf('Participant %d, Recording %d\n',Sbjs(i),j))
            F = load(['Data/HT' num2str(Sbjs(i)) '_0' num2str(j) '.mat']);
            HT = F.TrackVector;                 % Head tracking data
            HT = resample(HT,fs,fs_orig);       % Downsampling to 4410 Hz (/10)
            if ismember(i,RightEar_NH),     HT = -HT; end  % Flip RIGHT bad ear
            if ismember(i-16,RightEar_UHL), HT = -HT; end  % Flip RIGHT bad ear
            HT = HT(5*fs:end-5*fs);             % Remove first and last 5 secs
            h_ind(j,:) = histcounts(HT,edges,'Normalization','probability');
            STD((i-1)*6+j,1) = i;
            STD((i-1)*6+j,2) = std(HT);
    
            % figure control
            if DEBUG
                clf(figure(1))                                         %#ok
                histogram('BinEdges',edges,'BinCounts',h_ind(j,:))
                title(sprintf('Subject %d  |  Run %d',i,j))
                pause
            end

            figure(2)
            t = 0:1/fs:(length(HT)-1)/fs;
            plot(t,HT)
            ylim([min(edges) max(edges)])
            title(sprintf('Subject %d  |  Run %d',i,j))
            if DEBUG, pause; end                                       %#ok
        end
    end

    % % Get averaged histogram per participant
    H(i,:) = nanmean(h_ind);                                           %#ok
    
    clf(figure(3)),hold on
    if sum(isfinite(H(i,:)))==L_hist
        histogram('BinEdges',edges,'BinCounts',H(i,:))
    end
    title(sprintf('Subject %d',i))
    if DEBUG, pause; end                                               %#ok
end

%% Store averaged histograms
H_NH = nanmean(H(1:16,:));                                             %#ok
H_UHL = nanmean(H(17:32,:));                                           %#ok
save('Averaged_Histograms','H_NH','H_UHL','STD')
close all